/*
 * Copyright (c) 2016, Freescale Semiconductor, Inc.
 * Copyright 2016-2019 NXP
 * All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*  Standard C Included Files */
#include <stdio.h>
#include <string.h>
#include "board.h"
#include "fsl_debug_console.h"
#include "fsl_i2c.h"
#include "fsl_mrt.h"
#include "fsl_ctimer.h"
#include "pin_mux.h"
#include "fsl_adc.h"
#include "fsl_spi.h"
#include "fsl_power.h"
#include "math.h"

#include "lcd.h"
#include "lcd_fonts.h"
/*******************************************************************************
 * Definitions
 ******************************************************************************/
//I2C define
#define EXAMPLE_I2C_MASTER_BASE (I2C1_BASE)
#define I2C_MASTER_CLOCK_FREQUENCY CLOCK_GetMainClkFreq()
#define WAIT_TIME 10U
#define EXAMPLE_I2C_MASTER ((I2C_Type *)EXAMPLE_I2C_MASTER_BASE)
#define I2C_MASTER_SLAVE_ADDR_7BIT 0x0BU
#define I2C_BAUDRATE 200000U
#define I2C_DATA_LENGTH 3U
#define I2C_DATA_WRITE_LENGTH 2U
//smart battery register define
#define Temperature_Add 0x08U
#define Voltage_Add 0x09U
#define Current_Add 0x0AU
#define RemainingCapacity_Add 0x0DU
#define DischargeRemainingTime_Add 0x11U
#define ChargeRemainingTime_Add 0x13U
//PWM time define
#define MRT_CLK_FREQ CLOCK_GetFreq(kCLOCK_CoreSysClk)
#define CTIMER CTIMER0                 /* Timer 0 */
#define CTIMER_MAT_OUT kCTIMER_Match_1 /* Match output 1 */
#define CTIMER_CLK_FREQ CLOCK_GetFreq(kCLOCK_CoreSysClk)
//NTC temperature sensor from battery, ADC define
#define DEMO_ADC_BASE ADC0
#define DEMO_ADC_SAMPLE_CHANNEL_NUMBER 0U
#define DEMO_ADC_CLOCK_SOURCE kCLOCK_Fro
#define DEMO_ADC_CLOCK_DIVIDER 1U
//LCD display/SPI define
#define APP_LCD_DEMO_TITLE_BASE_LINE_IDX  1u
#define EXAMPLE_SPI_MASTER SPI0
#define EXAMPLE_CLK_SRC kCLOCK_MainClk
#define EXAMPLE_SPI_MASTER_CLK_FREQ CLOCK_GetFreq(EXAMPLE_CLK_SRC)
#define EXAMPLE_SPI_MASTER_BAUDRATE 1000000U
#define EXAMPLE_SPI_MASTER_SSEL kSPI_Ssel0Assert

/*******************************************************************************
 * Prototypes
 ******************************************************************************/
/* ADC. */
static void ADC_Configuration(void);
/* LCD. */
void lcd_print_string(uint16_t line_idx, char *buf, uint16_t hwColor);
void lcd_print_string_add_row(uint16_t line_idx, uint16_t row_idx, char *buf, uint16_t hwColor);
/*******************************************************************************
 * Variables
 ******************************************************************************/
//smart battery result
uint16_t g_Voltage;
uint16_t g_Current;
uint16_t g_Temperature;
uint16_t g_RemainingCapacity;
uint16_t g_ChargeRemainingTime;

//samrt battery threshold between stages
uint16_t g_PreChargeMaxVoltage;
uint16_t g_CCChargeMaxVoltage;
uint16_t g_CVChargeMinCurrent;



//polling cycles
static volatile uint32_t g_mrtCountValue     = 0u;
static volatile uint32_t g_DutyDelayStage1   = 0u;
static volatile uint32_t g_DutyDelayStage2   = 0u;
//pwm configure
volatile uint32_t g_pwmPeriod   = 0U;
volatile uint32_t g_pulsePeriod = 0U;
volatile uint32_t g_timerClock  = 0U;
volatile uint32_t g_cycleduty   = 0U;
//charging status
typedef enum Status
{
    initial = 0U, 	/*battery is pre-charging */
    Pre_Charge , 	/*battery is pre-charging */
    CC_Charge,	 			/*battery is constant current charging */
    CV_Charge,       	/*battery is constant voltage charging */
    Charge_Full,      /*battery is charged full */
    Over_Current,	 		/*current is above 500mA */
    Over_Voltage,	 		/*voltage is above 8200mV */
    Over_Temperature,	/*temperature is above 50 */
		Disconnect,				/*deivce is disconnected */
} ChargeStatus;

int8_t g_ChargeStatus;
//ADC 
adc_result_info_t adcResultInfoStruct;
const uint32_t g_Adc_12bitFullRange = 4096U;
const float Rp = 10000.0;
const float T2 = (273.15+25.0);
const float Bx =	3950.0;
uint32_t g_Temperature_NTC;
float g_Voltage_NTC;
float Rt;
//LCD display
char gLcdPrintfBuf[32];
char gLcdPrintfBack[240] ={0};
/*******************************************************************************
 * Code
 ******************************************************************************/
status_t CTIMER_GetPwmPeriodValue(uint32_t pwmFreqHz, uint8_t dutyCyclePercent, uint32_t timerClock_Hz)
{
    /* Calculate PWM period match value */
    g_pwmPeriod = (timerClock_Hz / pwmFreqHz) - 1;

    /* Calculate pulse width match value */
    if (dutyCyclePercent == 0)
    {
        g_pulsePeriod = g_pwmPeriod + 1;
    }
    else
    {
        g_pulsePeriod = (g_pwmPeriod * (500 - dutyCyclePercent)) / 500;
    }
    return kStatus_Success;
}
 
/*!
 * @brief read vaule of battery function
 */
uint16_t Information_Read_From_Battery(uint32_t address){
	
		status_t reVal  = kStatus_Fail;
		uint8_t deviceAddress;
		uint8_t  RxData[2];
		uint16_t ReturnData;

		deviceAddress = address;	
		if (kStatus_Success == I2C_MasterStart(EXAMPLE_I2C_MASTER, I2C_MASTER_SLAVE_ADDR_7BIT, kI2C_Write))
		{
				reVal = I2C_MasterWriteBlocking(EXAMPLE_I2C_MASTER, &deviceAddress, 1, kI2C_TransferNoStopFlag);
				if (reVal != kStatus_Success)
				{
						return -1;
				}
				reVal = I2C_MasterRepeatedStart(EXAMPLE_I2C_MASTER, I2C_MASTER_SLAVE_ADDR_7BIT, kI2C_Read);
				if (reVal != kStatus_Success)
				{
						return -1;
				}
				reVal = I2C_MasterReadBlocking(EXAMPLE_I2C_MASTER, RxData, I2C_DATA_LENGTH - 1, kI2C_TransferDefaultFlag);
				if (reVal != kStatus_Success)
				{
						return -1;
				}
				reVal = I2C_MasterStop(EXAMPLE_I2C_MASTER);
				if (reVal != kStatus_Success)
				{
						return -1;
				}
				ReturnData = (RxData[0]) |(RxData[1]<<8);		
		}
		return(ReturnData);
}
/*!
 * @brief timer interrupt handler
 */
void MRT0_IRQHandler(void)
{
    /* Clear interrupt flag.*/
    MRT_ClearStatusFlags(MRT0, kMRT_Channel_0, kMRT_TimerInterruptFlag);
		g_mrtCountValue++;  //increatment every 100ms
	
		if((g_mrtCountValue% 4) == 0){ //every 500ms	
			
			g_Voltage = Information_Read_From_Battery(Voltage_Add);	
			g_Current = Information_Read_From_Battery(Current_Add);
//			PRINTF("g_Current: %d \r\n" , g_Current);
//			PRINTF("g_Voltage: %d \r\n" , g_Voltage);	
//			PRINTF("g_cycleduty: %d \r\n" , g_cycleduty);
//			PRINTF("g_ChargeStatus: %d \r\n" , g_ChargeStatus);			
			if((g_Voltage > g_PreChargeMaxVoltage)&&(g_ChargeStatus == Pre_Charge)){
					g_ChargeStatus = CC_Charge;
				  g_DutyDelayStage1=0;
			}	
			switch (g_ChargeStatus) {
				case Pre_Charge:  
					g_cycleduty =  35;
//					PRINTF("Pre_Charge \r\n" );			
					GPIO_PortClear(GPIO, 0, 1u << 24); //blue LED
					GPIO_PortSet(GPIO, 0, 1u << 25);				
					GPIO_PortSet(GPIO, 0, 1u << 26);	
					break;
				case CC_Charge:  
//					PRINTF("CC_Charge \r\n" );			
					if(g_Voltage > g_CCChargeMaxVoltage){
						g_ChargeStatus = CV_Charge;
					}		
					GPIO_PortClear(GPIO, 0, 1u << 24); //blue LED
					GPIO_PortSet(GPIO, 0, 1u << 25);				
					GPIO_PortSet(GPIO, 0, 1u << 26);	
					if((g_Current >= 0)&&(g_Current < 80)){	
						//configure per every 50s
						if(g_DutyDelayStage1%100 == 0){
							g_cycleduty =  100;		
							g_DutyDelayStage2=0;
						}	
						g_DutyDelayStage1++;
					}else if((g_Current >=80)&& (g_Current < 200)){
						//configure per every 50s
						if(g_DutyDelayStage2%100 == 0){
							g_cycleduty =  180;											
						}						
						g_DutyDelayStage2++;						
					}else if((g_Current >=200)&&(g_Current < 345)){
						g_cycleduty++;				
					}else if((g_Current >=345)&&(g_Current < 355)){
		
					}else if((g_Current >=355)&&(g_Current < 500)){
						g_cycleduty--;
					}else if(g_Current >= 500){	
						g_ChargeStatus = Over_Current;
						g_cycleduty = 0;		
						GPIO_PortClear(GPIO, 0, 1u << 26); //red LED
						GPIO_PortSet(GPIO, 0, 1u << 24);				
						GPIO_PortSet(GPIO, 0, 1u << 25);									
					}	

					break;				
				case CV_Charge:

//					PRINTF("CV_Charge \r\n" );
					if(g_Current < g_CVChargeMinCurrent){
						g_ChargeStatus = Charge_Full;
					}			
					GPIO_PortClear(GPIO, 0, 1u << 24); //blue LED
					GPIO_PortSet(GPIO, 0, 1u << 25);				
					GPIO_PortSet(GPIO, 0, 1u << 26);					
					if((g_Voltage >g_CCChargeMaxVoltage - 50)&&(g_Voltage <= g_CCChargeMaxVoltage)){					
							g_cycleduty++;
					}else if((g_Voltage >g_CCChargeMaxVoltage)&&(g_Voltage <= g_CCChargeMaxVoltage+5)){
						
					}else if((g_Voltage >g_CCChargeMaxVoltage+5)&&(g_Voltage <= g_CCChargeMaxVoltage+50)){
							g_cycleduty--;				
					}else if(g_Voltage >g_CCChargeMaxVoltage+50){
						g_ChargeStatus = Over_Voltage;
						g_cycleduty = 0;		
						GPIO_PortClear(GPIO, 0, 1u << 26); //red LED
						GPIO_PortSet(GPIO, 0, 1u << 24);				
						GPIO_PortSet(GPIO, 0, 1u << 25);			
					}else{			
						g_cycleduty =  200;
					}	
					break;
				case Charge_Full:		
//					PRINTF("Charge_Full \r\n" );								
					g_ChargeStatus = Charge_Full;					
					g_cycleduty = 0;	
					GPIO_PortClear(GPIO, 0, 1u << 25); //green LED
					GPIO_PortSet(GPIO, 0, 1u << 24);				
					GPIO_PortSet(GPIO, 0, 1u << 26);				
					break;
			}	
			CTIMER_GetPwmPeriodValue(70000, g_cycleduty, g_timerClock);
			CTIMER_SetupPwmPeriod(CTIMER, CTIMER_MAT_OUT, g_pwmPeriod, g_pulsePeriod, false);		
		
//			g_Temperature = Information_Read_From_Battery(Temperature_Add);				
			ADC_DoSoftwareTriggerConvSeqA(DEMO_ADC_BASE);
			/* Wait for the converter to be done. */
			while (!ADC_GetChannelConversionResult(DEMO_ADC_BASE, DEMO_ADC_SAMPLE_CHANNEL_NUMBER, &adcResultInfoStruct))
			{
			}
			g_Voltage_NTC = (float)adcResultInfoStruct.result/4096*3.3;
			Rt = (33200*g_Voltage_NTC)/(3.3 - g_Voltage_NTC);
			g_Temperature_NTC = (1/(log(Rt/Rp)/Bx+(1/T2)))-273.15;	
			
			if((g_Temperature_NTC >50)&&(g_Temperature_NTC <=70)){  //over temperautre protect
				g_ChargeStatus = Over_Temperature;		
				CTIMER_StopTimer(CTIMER);
				GPIO_PortClear(GPIO, 0, 1u << 26);//red LED
				GPIO_PortSet(GPIO, 0, 1u << 25);
				GPIO_PortSet(GPIO, 0, 1u << 24);				
			}	else if(g_Temperature_NTC > 70){  //over temperautre protect
				g_ChargeStatus = Disconnect;	
				CTIMER_StopTimer(CTIMER);
				GPIO_PortClear(GPIO, 0, 1u << 26);//red LED
				GPIO_PortSet(GPIO, 0, 1u << 25);
				GPIO_PortSet(GPIO, 0, 1u << 24);				
			}
		}	

		if((g_mrtCountValue% 20) == 0){  //every 2S
			lcd_clear_screen(LCD_COLOR_BLACK);			
			g_ChargeRemainingTime = Information_Read_From_Battery(ChargeRemainingTime_Add);	
			g_RemainingCapacity   = Information_Read_From_Battery(RemainingCapacity_Add);			
			if((g_ChargeStatus == CC_Charge)||(g_ChargeStatus == CV_Charge)){
				sprintf(gLcdPrintfBuf, "Charging!!!");
				sprintf(gLcdPrintfBuf, "Charging!!!");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+2u, gLcdPrintfBuf,  LCD_COLOR_GREEN);
				sprintf(gLcdPrintfBuf, "Remaining time:");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+8u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
				sprintf(gLcdPrintfBuf, "%d min", g_ChargeRemainingTime);
				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+8,150, gLcdPrintfBuf, LCD_COLOR_YELLOW);	
				
				PRINTF("Charging!!!Remaining time: %d min\r\n" , g_ChargeRemainingTime);	

			}else if(g_ChargeStatus == Charge_Full){
				sprintf(gLcdPrintfBuf, "Charged full");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+2u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);			
				PRINTF("Charged full\r\n");	
				
			}else if(g_ChargeStatus == Pre_Charge){
				sprintf(gLcdPrintfBuf, "PreCharging");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+2u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);	
				PRINTF("PreCharging\r\n");	
				
			}else if(g_ChargeStatus == Over_Temperature){
				sprintf(gLcdPrintfBuf, "Over Temperature!!!");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+2u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);			
				PRINTF("Over Temperature!!!\r\n");	
				
			}
			
			if(g_ChargeStatus == Disconnect){
				sprintf(gLcdPrintfBuf, "Please connect battery!!!");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+2u, gLcdPrintfBuf,  LCD_COLOR_RED);	
				sprintf(gLcdPrintfBuf, "And re-power the board!!!");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+3u, gLcdPrintfBuf,  LCD_COLOR_RED);	

				PRINTF("Please connect battery!!!\r\n");	
				PRINTF("And re-power the board!!!\r\n");	
				
			}else{
				sprintf(gLcdPrintfBuf, "Voltage:");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+5u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
				sprintf(gLcdPrintfBuf, "%d mV\r\n", g_Voltage);
				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+5,150, gLcdPrintfBuf, LCD_COLOR_YELLOW);
				
				sprintf(gLcdPrintfBuf, "Current:");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+6u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
				sprintf(gLcdPrintfBuf, "%d mA\r\n",  g_Current);
				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+6,150, gLcdPrintfBuf, LCD_COLOR_YELLOW);	
				
				sprintf(gLcdPrintfBuf, "Temperature:");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+4u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
				sprintf(gLcdPrintfBuf, "%d degree\r\n", (uint32_t)g_Temperature_NTC);
				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+4,150, gLcdPrintfBuf, LCD_COLOR_YELLOW);		
						
				sprintf(gLcdPrintfBuf, "Remaining capacity:");
				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+7u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
				sprintf(gLcdPrintfBuf, "%d %\r\n", g_RemainingCapacity);
				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+7,150, gLcdPrintfBuf, LCD_COLOR_YELLOW);	
				PRINTF("Voltage: %d mV\r\n" , g_Voltage);
				PRINTF("Current: %d mA\r\n" , g_Current);	
				PRINTF("Temperature: %d degree\r\n" ,(uint32_t) g_Temperature_NTC);	
				PRINTF("Remaining capacity: %d %\r\n" ,g_RemainingCapacity);	
	      PRINTF("\r\n\r\n" );			
		
//				sprintf(gLcdPrintfBuf, "PWMcycleduty:");
//				lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+9u, gLcdPrintfBuf,  LCD_COLOR_BLUE | LCD_COLOR_RED);				
//				sprintf(gLcdPrintfBuf, " %d/500 percent\r\n", g_cycleduty);
//				lcd_print_string_add_row(APP_LCD_DEMO_TITLE_BASE_LINE_IDX+9,100, gLcdPrintfBuf, LCD_COLOR_YELLOW);
			}					
		}					
		if(g_mrtCountValue > 99){
				g_mrtCountValue = 0;
		}

}

/*!
 * @brief Main function
 */
int main(void)
{
    i2c_master_config_t masterConfig;
    status_t reVal        = kStatus_Fail;
    uint8_t deviceAddress = 0x04U;
	  uint32_t mrt_clock;
    /* Structure of initialize MRT */
    mrt_config_t mrtConfig;
	
	  ctimer_config_t config;
    uint32_t srcClock_Hz;
    uint32_t frequency = 0U;
		/* Define the init structure for the output LED pin*/
    gpio_pin_config_t led_config = {
        kGPIO_DigitalOutput,
        0,
    };
	
	   /* Attach FRO clock to ADC0. */
    CLOCK_Select(kADC_Clk_From_Fro);
    CLOCK_SetClkDivider(kCLOCK_DivAdcClk, 1U);
    /* Power on ADC0. */
    POWER_DisablePD(kPDRUNCFG_PD_ADC0);
    /* Select the main clock as source clock of USART0 (debug console) */
    CLOCK_Select(BOARD_DEBUG_USART_CLK_ATTACH);
    CLOCK_Select(kI2C1_Clk_From_MainClk);
    /* Attach main clock to SPI0. */
    CLOCK_Select(kSPI0_Clk_From_MainClk);
		
		
    BOARD_InitPins();
    BOARD_BootClockFRO30M();
    BOARD_InitDebugConsole();
    PRINTF("\r\n Smart charger demo\r\n ");
		//initiallize LED pins
    GPIO_PortInit(GPIO, 0);
    GPIO_PinInit(GPIO, 0, 24, &led_config);
    GPIO_PinInit(GPIO, 0, 25, &led_config);
    GPIO_PinInit(GPIO, 0, 26, &led_config);
    GPIO_PortSet(GPIO, 0, 1u << 24);
    GPIO_PortSet(GPIO, 0, 1u << 25);
    GPIO_PortSet(GPIO, 0, 1u << 26);
	
		//initiallize P1_19 and 1_20 for thermistor power
    GPIO_PortInit(GPIO, 1);
    GPIO_PinInit(GPIO, 1, 19, &led_config);
    GPIO_PinInit(GPIO, 1, 20, &led_config);
    GPIO_PortSet(GPIO, 1, 1u << 19);
    GPIO_PortClear(GPIO, 1, 1u << 20);

		//initiallize CTimer0 to generate PWM waves for bulk convertor
    /* CTimer0 counter uses the AHB clock, some CTimer1 modules use the Aysnc clock */
    srcClock_Hz = CTIMER_CLK_FREQ;
    CTIMER_GetDefaultConfig(&config);
    g_timerClock = srcClock_Hz / (config.prescale + 1);
    CTIMER_Init(CTIMER, &config);
    /* Get the PWM period match value and pulse width match value of 20Khz PWM signal with 20% dutycycle */
    CTIMER_GetPwmPeriodValue(70000, 1, g_timerClock);
    CTIMER_SetupPwmPeriod(CTIMER, CTIMER_MAT_OUT, g_pwmPeriod, g_pulsePeriod, false);
    CTIMER_StartTimer(CTIMER);


    frequency = CLOCK_GetFreq(DEMO_ADC_CLOCK_SOURCE) / CLOCK_GetClkDivider(kCLOCK_DivAdcClk);
    if (true == ADC_DoSelfCalibration(DEMO_ADC_BASE, frequency))
    {
//        PRINTF("ADC Calibration Done.\r\n");
    }
    else
    {
//        PRINTF("ADC Calibration Failed.\r\n");
    }
    /* Configure the converter and work mode. */
    ADC_Configuration();
		
		
		//initiallize LCD screen	
    lcd_init();
    lcd_clear_screen(LCD_COLOR_BLACK);	
		lcd_print_string(APP_LCD_DEMO_TITLE_BASE_LINE_IDX, "Smart Charger Demo", LCD_COLOR_GREEN);
		
		
		/*
     * masterConfig.debugEnable = false;
     * masterConfig.ignoreAck = false;
     * masterConfig.pinConfig = kI2C_2PinOpenDrain;
     * masterConfig.baudRate_Bps = 100000U;
     * masterConfig.busIdleTimeout_ns = 0;
     * masterConfig.pinLowTimeout_ns = 0;
     * masterConfig.sdaGlitchFilterWidth_ns = 0;
     * masterConfig.sclGlitchFilterWidth_ns = 0;
     */
    I2C_MasterGetDefaultConfig(&masterConfig);
    /* Change the default baudrate configuration */
    masterConfig.baudRate_Bps = I2C_BAUDRATE;
    /* Initialize the I2C master peripheral */
    I2C_MasterInit(EXAMPLE_I2C_MASTER, &masterConfig, I2C_MASTER_CLOCK_FREQUENCY);


		g_ChargeStatus = Pre_Charge;  // the default stage is pre-charge mode.
		g_PreChargeMaxVoltage = 6200;	//6.2V, if above g_PreChargeMaxVoltage it will enter CC charge mode.
		g_CVChargeMinCurrent = 40;		//40mA, if above g_PreChargeMaxVoltage it will enter CV charge mode.
		g_CCChargeMaxVoltage = 8150;	//8.15V if above g_PreChargeMaxVoltage it will enter charge full mode.
		
		//generate 100ms peirodic interrupt
		mrt_clock = MRT_CLK_FREQ;
    /* mrtConfig.enableMultiTask = false; */
    MRT_GetDefaultConfig(&mrtConfig);
    /* Init mrt module */
    MRT_Init(MRT0, &mrtConfig);
    /* Setup Channel 0 to be repeated */
    MRT_SetupChannelMode(MRT0, kMRT_Channel_0, kMRT_RepeatMode);
    /* Enable timer interrupts for channel 0 */
    MRT_EnableInterrupts(MRT0, kMRT_Channel_0, kMRT_TimerInterruptEnable);
    /* Enable at the NVIC */
    EnableIRQ(MRT0_IRQn);
    /* Start channel 0 */
    MRT_StartTimer(MRT0, kMRT_Channel_0, USEC_TO_COUNT(100000U, mrt_clock)); //max time is 70s

		
		
		
    while (1)
    {

			
    }
}
/*!
 * @brief ADC configuration
 */
static void ADC_Configuration(void)
{
    adc_config_t adcConfigStruct;
    adc_conv_seq_config_t adcConvSeqConfigStruct;

/* Configure the converter. */
                                                   /* FSL_FEATURE_ADC_HAS_CTRL_ASYNMODE */
    adcConfigStruct.clockDividerNumber = DEMO_ADC_CLOCK_DIVIDER;

#if defined(FSL_FEATURE_ADC_HAS_CTRL_LPWRMODE) & FSL_FEATURE_ADC_HAS_CTRL_LPWRMODE
    adcConfigStruct.enableLowPowerMode = false;
#endif /* FSL_FEATURE_ADC_HAS_CTRL_LPWRMODE */
#if defined(FSL_FEATURE_ADC_HAS_TRIM_REG) & FSL_FEATURE_ADC_HAS_TRIM_REG
    adcConfigStruct.voltageRange = kADC_HighVoltageRange;
#endif /* FSL_FEATURE_ADC_HAS_TRIM_REG */
    ADC_Init(DEMO_ADC_BASE, &adcConfigStruct);
    /* Enable channel DEMO_ADC_SAMPLE_CHANNEL_NUMBER's conversion in Sequence A. */
    adcConvSeqConfigStruct.channelMask =
        (1U << DEMO_ADC_SAMPLE_CHANNEL_NUMBER); /* Includes channel DEMO_ADC_SAMPLE_CHANNEL_NUMBER. */
    adcConvSeqConfigStruct.triggerMask      = 0U;
    adcConvSeqConfigStruct.triggerPolarity  = kADC_TriggerPolarityPositiveEdge;
    adcConvSeqConfigStruct.enableSingleStep = false;
    adcConvSeqConfigStruct.enableSyncBypass = false;
    adcConvSeqConfigStruct.interruptMode    = kADC_InterruptForEachSequence;
    ADC_SetConvSeqAConfig(DEMO_ADC_BASE, &adcConvSeqConfigStruct);
    ADC_EnableConvSeqA(DEMO_ADC_BASE, true); /* Enable the conversion sequence A. */
    /* Clear the result register. */
    ADC_DoSoftwareTriggerConvSeqA(DEMO_ADC_BASE);
    while (!ADC_GetChannelConversionResult(DEMO_ADC_BASE, DEMO_ADC_SAMPLE_CHANNEL_NUMBER, &adcResultInfoStruct))
    {
    }
    ADC_GetConvSeqAGlobalConversionResult(DEMO_ADC_BASE, &adcResultInfoStruct);
}

/*!
 * @brief LCD print string 
 */
void lcd_print_string(uint16_t line_idx, char *buf, uint16_t hwColor)
{
    uint16_t y = line_idx * 16u;
    uint16_t x = 0u;

    while (x < (LCD_WIDTH-16u) )
    {
        lcd_display_char(x, y, *buf++, LCD_FONT_1608, hwColor);
        if (*buf == '\0')
        {
            break;
        }
        else if ( (*buf == '\r') || (*buf == '\n') )
        {
            break;
        }
        else
        {
            x += 8u;
        }
    }
}

/*!
 * @brief print string add row
 */
void lcd_print_string_add_row(uint16_t line_idx,uint16_t row_idx, char *buf, uint16_t hwColor)
{
    uint16_t y = line_idx * 16u;
    uint16_t x = row_idx;

    while (x < (LCD_WIDTH-16u) )
    {
        lcd_display_char(x, y, *buf++, LCD_FONT_1608, hwColor);
        if (*buf == '\0')
        {
            break;
        }
        else if ( (*buf == '\r') || (*buf == '\n') )
        {
            break;
        }
        else
        {
            x += 8u;
        }
    }
}

